import {v4 as uuidv4} from 'uuid';
import supabaseClient from './supabaseClient';

// The back-end database uses snake_case column names, while this goes against the conventions
// we used during the datamodelling course last year, is is regretfully required.
// If camelCase variables are used, the backend won't accept access policies because the
// camelCase variabels get converted to camelcase for some reason...

/**
 * Add a new To-Do list to the backend database.
 * @param name The name of the new To-Do list;
 * @param is_private If true, the list is private and can't be edited or viewed by other users.
 * @return {Promise<undefined|string>} This method either returns undefined (on errors) or the uuid of the new list.
 */
export const createList = async (name, is_private) => {
    const uuid = uuidv4();
    try {
        const user = supabaseClient.auth.user();
        let {error} = await supabaseClient
            .from('lists')
            .insert([{uuid, name, user_id: user.id, is_private}])
        if (error) {
            throw error;
        }
        return uuid;
    } catch (error) {
        console.log('error', error)
    }
    return undefined;
}

/**
 * Retrieve a specific list from the backend database.
 * @param id The is of the list that is to be retrieved.
 * @param abortController An optional AbortController instance.
 * @return {Promise<undefined|any>} If the list could be retrieved it is returned, otherwise undefined will be returned.
 */
export const fetchList = async (id, abortController = undefined) => {
    abortController = abortController ? abortController : new AbortController();
    try {
        let {data, error} = await supabaseClient
            .from('lists')
            .select(`
                *, 
                user:user_id(username, id), 
                tasks(*, createdBy:created_by(username, id), 
                completedBy:completed_by(username, id))
            `)
            .eq('uuid', id)
            .abortSignal(abortController.signal)
            .single()

        if (error) {
            throw error
        }

        data.tasks.sort((a, b) => a.id > b.id);
        return data;
    } catch (error) {
        console.error(`The following error occurred while retrieving the list with id ${id}`);
        console.error(error);
    }
    return undefined;
}

/**
 * Retrieve all the list that were created by the current user.
 * @param abortController An optional AbortController instance.
 * @return {Promise<any[]|*[]>} If the user's lists couldn't be retrieved an empty array is returned,
 * otherwise the lists are returned.
 */
export const fetchAllListCreatedByUser = async (abortController) => {
    abortController = abortController ? abortController : new AbortController();

    try {
        const user = supabaseClient.auth.user();
        let {data, error} = await supabaseClient
            .from('lists')
            .select('*, user:user_id(username)')
            .eq('user_id', user.id)
            .abortSignal(abortController.signal)
        if (error) {
            throw error;
        }

        return data
    } catch (error) {
        console.error(`The following error occurred while trying to fetch all lists created by a specific user:`);
        console.error(error);
    }
    return [];
}

/**
 * Retrieve all the public lists.
 * @param abortController An optional AbortController instance.
 * @return {Promise<any[]|*[]>} If the public lists couldn't be retrieved an empty array is returned,
 * otherwise the lists are returned.
 */
export const fetchAllPublicLists = async (abortController) => {
    abortController = abortController ? abortController : new AbortController();

    try {
        let {data, error} = await supabaseClient
            .from('lists')
            .select('*, user:user_id(username)')
            .eq('is_private', false)
            .abortSignal(abortController.signal)

        if (error) {
            throw error;
        }

        return data
    } catch (error) {
        console.error('The following error occurred while trying to fetch all public lists: ');
        console.error(error);
    }
    return [];
}

/**
 * Update the status of a specific task in a specific list.
 * @param list_id The id of the list that contains the task.
 * @param task_id The id of the task for which the status must be changed.
 * @param complete The new status of the task.
 * @return {Promise<boolean>} True if the request completed without error, false otherwise.
 */
export const updateTask = async (list_id, task_id, complete) => {
    try {
        const userId = supabaseClient.auth.user().id;
        let updates = {
            id: task_id,
            list_id,
            complete,
            completed_by: complete ? userId : null,
        }

        let {error} = await supabaseClient
            .from('tasks')
            .update(updates, {returning: 'minimal'})
            .eq('id', task_id)

        if (error) {
            throw error;
        }
        return true;
    } catch (error) {
        console.error(`An error occurred while updating the task with id ${task_id} in the list with id ${list_id}`);
        console.error(error);
    }
    return false;
}

/**
 * Create a new task in a given list.
 * @param name The name of the new task.
 * @param list_id The id of the list in which the new task must be added.
 * @return {Promise<boolean>} True if the request completed without error, false otherwise.
 */
export const createTask = async (name, list_id) => {
    try {
        const userId = supabaseClient.auth.user().id;
        const task = {
            list_id,
            complete: false,
            created_by: userId,
            name
        }

        let {error} = await supabaseClient
            .from('tasks')
            .insert(task, {returning: 'minimal'})

        if (error) {
            throw error;
        }
        return true;
    } catch (error) {
        console.error(`An error occurred while inserting a new task in the list with id ${list_id}`);
        console.error(error);
    }
    return false;
}

/**
 * Delete the task with the given id.
 * @param id The id of the task which must be deleted.
 * @return {Promise<boolean>} True if the request completed without error, false otherwise.
 */
export const deleteTask = async (id) => {
    try {
        let {error} = await supabaseClient
            .from('tasks')
            .delete()
            .eq('id', id);

        if (error) {
            throw error;
        }
        return true;
    } catch (error) {
        console.error(`An error occurred while trying to delete the task with id: ${id}`);
        console.error(error);
    }
    return false;
}
