import {Button, Col, Form, ListGroup, Row} from 'react-bootstrap';
import {useRef, useState} from 'react';
import {useParams} from 'react-router-dom';
import TaskItem from './taskItem';
import useProfile from '../../hooks/useProfile';
import useToDoStore from '../../hooks/useToDoStore';

const ToDoListDetail = () => {
    const [newTaskName, setNewTaskName] = useState('');
    const createButton = useRef(null);

    /**
     * useProfile is a custom hook that retrieves the username and id of the currently logged in user.
     */
    const {userId} = useProfile()

    /**
     * The useParams hook is included in react-router and can be used to access the navigation parameters, i.e.
     * the parameters passed through the URL.
     */
    const {id} = useParams();

    /**
     * The useToDoStore hook is a custom hook which retrieves the contents of a single To-Do list.
     */
    const [
        toDoList,
        changeTaskStatus,
        deleteTaskFromList,
        addTaskToList] = useToDoStore(id);

    const createTaskHandler = async (event) => {
        event.preventDefault();
        createButton.current.blur();
        setNewTaskName('');
        await addTaskToList(newTaskName);
    }

    return <div>
        <Row>
            <Col>
                <h1>{toDoList?.name}</h1>
                <div className={'text-muted'}>
                    Created by {toDoList?.user?.username}
                </div>
                <hr/>
            </Col>
        </Row>

        <Row className={'mb-2 mt-3'}>
            <Form onSubmit={createTaskHandler}>
                <Row>
                    <Col sm={6}>
                        <Form.Group className="mb-3" controlId="formBasicEmail">
                            <Form.Control type="text" value={newTaskName} placeholder="New task"
                                          onChange={evt => setNewTaskName(evt.target.value)}/>
                        </Form.Group>
                    </Col>
                    <Col sm={6}>
                        <div className={'d-grid'}>
                            <Button variant="primary" type="submit" ref={createButton}>
                                Create task
                            </Button>
                        </div>
                    </Col>
                </Row>
            </Form>
        </Row>

        <Row>
            <Col>
                <ListGroup>
                    {toDoList
                        ?.tasks
                        ?.map(t => <TaskItem key={t.id} {...t}
                                             allowDelete={userId === t.createdBy.id || userId === toDoList.user.id}
                                             changeTaskStatus={() => changeTaskStatus(t.name, t.id, !t.complete)}
                                             deleteTaskHandler={() => deleteTaskFromList(t.id)}/>)}
                </ListGroup>
            </Col>
        </Row>
    </div>;
}

export default ToDoListDetail;
