import {useRef, useState} from 'react';
import {Button, Form} from 'react-bootstrap';
import {createList} from '../../api/toDoApi';
import {useHistory} from 'react-router-dom';

const ToDoListForm = () => {
    /**
     * The useState hook can be used to store data in function components that is persistent through multiple
     * renders. However, only data which influences the re-renders should be saved in a useState hook.
     *
     * State should NOT be saved as an object, see the ToDoListFormWithSingleState component for the reasons why.
     */
    const [listName, setListName] = useState('');
    const [isPrivate, setIsPrivate] = useState(false);

    /**
     * The useRef hook can be used to save references to specific DOM elements, and thus can be used
     * to modify the DOM directly. This should be avoided as much as possible.
     */
    const submitRef = useRef(null);

    /**
     * The useHistory hook provided by react-router can be used to navigate to a previous page, or to add
     * a new page to the navigation stack.
     */
    const history = useHistory();

    const submitForm = async (event) => {
        event.preventDefault();
        submitRef.current.blur();
        const uuid = await createList(listName, isPrivate);
        if (uuid) {
            history.push(`/lists/${uuid}`);
        }
    }

    return <Form onSubmit={submitForm}>
        <h1>New To-Do List</h1>
        <Form.Group className="mb-3">
            <Form.Label>To-Do list name</Form.Label>
            <Form.Control type="text" required placeholder="List Name" value={listName}
                          onChange={evt => setListName(evt.target.value)}/>
        </Form.Group>

        <Form.Group className="mb-3">
            <Form.Check type="checkbox" label="Is private"
                        checked={isPrivate} onChange={evt => setIsPrivate(evt.target.checked)}/>
        </Form.Group>

        <div className="d-grid gap-2">
            <Button variant="primary" type="submit"
                    ref={submitRef}>
                Create To-Do list
            </Button>
        </div>
    </Form>
}

export default ToDoListForm;
