import {fetchAllListCreatedByUser, fetchAllPublicLists} from '../../api/toDoApi';
import ToDoListItem from './toDoListItem';
import {useEffect, useState} from 'react';
import {Button, ListGroup, Tab, Tabs} from 'react-bootstrap';

const ToDoLists = () => {
    const [publicLists, setPublicLists] = useState([]);
    const [userLists, setUserLists] = useState([]);
    const [dependency, setDependency] = useState(0);

    /**
     * The useEffect hook is the ideal place to perform asynchronous operations such as API calls. Furthermore, it is
     * executed AFTER the first first render, and thus can also be used for any direct DOM manipulation.
     */
    useEffect(() => {
        const abortController = new AbortController();
        const fetchData = async () => {
            const lists = await fetchAllPublicLists(abortController);
            console.log("Fetched data in use effect, dependency array has changed.");

            /**
             * The fetchAllPublicLists function will return an empty array when the request fails or is cancelled,
             * thus we can prevent memory leaks by checking if the result is empty. If it is empty, this means one of two
             * things
             *
             *  1. The request failed and couldn't retrieve the required data: In this case, there is no point in updating
             *  the state since there is no data to show.
             *
             *  2. The request was cancelled by the cleanup function below: In this case, updating the state would
             *  result in a memory leak and should thus be avoided at all costs.
             */
            if (lists.length !== 0) setPublicLists(lists);
        }
        fetchData();

        /**
         * The return value of a useEffect call is a function that can be used to clean up any ongoing asynchronous
         * operations. This is required in most cases, as a state update after unmounting creates a memory leak.
         *
         * The cleanup function is run after EVERY call to the useEffect hook, and before unmounting. This means that,
         * if two consecutive state updates occur that require the useEffect hook to be called without the first call
         * having finished, that the first call will be cancelled and only the result of the second will used.
         */
        return () => abortController.abort();
    },
        /**
         * A useEffect hook can have an optional array of dependencies, this array determines when the useEffect call
         * will execute. As stated above, the default is a that useEffect hook will be executed after every render, this
         * is far from ideal, as this can result in multiple unnecessary API call and even in infinite loops. If a
         * single element in the array changes the hook will be executed again. Every variable that is used in the hook
         * (not assigned to but used as a parameter for some function) must be added to the dependency array.
         *
         * To ensure that a useEffect call is only executed after the first render and before unmounting, an empty array
         * can be passed as a dependency. Below we've added an array containing a single element, namely an integer value
         * dependency. Using the button below, this value can be incremented, and since it changed the hook will be
         * executed again.
         */
        [dependency]);


    useEffect(() => {
        const abortController = new AbortController();
        const fetchData = async () => {
            const lists = await fetchAllListCreatedByUser(abortController);
            if (lists.length !== 0) setUserLists(lists);
        }
        fetchData();

        return () => abortController.abort();
    }, []);


    return <div>
        <div className={'d-grid mb-4'}>
            <Button variant={'primary'}
                    onClick={(evt) => {
                        evt.target.blur();
                        setDependency((dependency) => dependency + 1)
                    }}>
                Increment dependencies &amp; trigger <i>useEffect</i>
            </Button>
        </div>

        <Tabs justify defaultActiveKey="public" id="uncontrolled-tab-example" className="mb-3">
            <Tab eventKey="public" title="Public lists">
                <h1>Public To-Do lists</h1>
                <ListGroup>
                    {publicLists.map(l => <ToDoListItem key={l.uuid} {...l}/>)}
                </ListGroup>
            </Tab>
            <Tab eventKey="user" title="My lists">
                <h2>My To-Do lists</h2>
                {userLists.map(l => <ToDoListItem key={l.uuid} {...l}/>)}
            </Tab>
        </Tabs>
    </div>
}

export default ToDoLists;
