import {useEffect, useRef, useState} from 'react';
import {createTask, deleteTask, fetchList, updateTask} from '../api/toDoApi';

/**
 * React allows us to write or own hooks. A hook is nothing but a function that performs some stateful logic.
 * The only requirement is that the function starts with the word use. Other than that, it can have as many parameters
 * as required and can return whatever it needs to.
 */
const useToDoStore = (id) => {
    const [toDoList, setToDoList] = useState({});
    const [shouldUpdate, setShouldUpdate] = useState(true);

    /**
     * The useRef hook can be used to add persistent variables to a function component, this seems somewhat hacky,
     * but is actually included in the official React docs.
     *
     * The reason useRef is required here, is that every non-state variable and function is re-created after a re-render.
     * While the variable or function might be exactly the same, it's object reference will have changed. So without
     * useRef the value of isMounted would not be reusable through multiple renders.
     */
    let isMounted = useRef(false);

    useEffect(() => {
        /**
         * Should update is required because we need a way to trigger a refresh of the To-Do list when a task has been
         * added, removed or updated.
         *
         * Since shouldUpdate is included in the dependency array, we need to check if it is true and only continue if
         * it is. If we don't do this, every request will be executed twice. The first time because the value changes
         * form the default (false) to true and than changes back to false when the update is completed.
         */
        if (!shouldUpdate) return;

        const abortController = new AbortController();
        const fetchData = async () => {
            const toDoList = await fetchList(id, abortController);
            if (toDoList) {
                setToDoList(toDoList);
                setShouldUpdate(false);
            }
            // Log so that the datamodel is visible in the console and so
            // that we can check the number of times the fetchData method is called.
            console.log(toDoList);
        }
        fetchData();
        return () => abortController.abort();
    }, [id, shouldUpdate])

    /**
     * This hook has an empty dependency array and is thus only executed after the first render and before unmount.
     */
    useEffect(() => {
        isMounted.current = true;
        return () => {isMounted.current = false};
    }, [])

    const changeTaskStatus = async (name, taskId, newStatus) => {
        const success = await updateTask(toDoList.id, taskId, newStatus);
        // We need isMounted here, because if the updateTask call takes a while the component might have unmounted
        // before it completes. If you then use the setShouldUpdate function, a memory leak will be created.
        // For those trying to reproduce the memory leak, it doesn't always happen. You might have to try a few different
        // times before you see the error message. But if you add a timeout to the updateTask method, you should be able
        // to reproduce the error, even if you can't do so consistently.
        if (success && isMounted.current) setShouldUpdate(true);
    }

    const deleteTaskFromList = async (taskId) => {
        const success = await deleteTask(taskId);
        if (success && isMounted.current) setShouldUpdate(true);
    }

    const addTaskToList = async (name) => {
        const success = await createTask(name, toDoList.id);
        if (success && isMounted.current) setShouldUpdate(true);
    }

    return [toDoList, changeTaskStatus, deleteTaskFromList, addTaskToList];
}

export default useToDoStore;
