import supabaseClient from './supabaseClient';

/**
 * Log in and send a confirmation link to the given email adres.
 * @param email The email adres that will be used to identify the user.
 * @return {Promise<boolean>}
 */
export const login = async (email) => {
    try {
        const {error} = await supabaseClient.auth.signIn({email});

        if (!error) {
            return true;
        }

    } catch (error) {
        console.error('The following error occurred while trying to log in:');
        console.error(error);
    }
    return false;
}

/**
 * Log out.
 * @return {Promise<void>}
 */
export const signOut = async () => {
    await supabaseClient.auth.signOut();
}

/**
 * Retrieve the profile and id of the currently logged in user.
 * @return {Promise<(string|any)[]|*[]>} Either an array of [userId, username] or [undefined, undefined]
 * when something went wrong.
 */
export const getProfile = async (abortController = undefined) => {
    abortController = abortController || new AbortController();
    try {
        const user = supabaseClient.auth.user();
        if (!user) {
            throw new Error('The profile couldn\'t be retrieved because the session data couldn\'t be read.');
        }

        let {data, error, status} = await supabaseClient
            .from('profiles')
            .select(`username`)
            .eq('id', user.id)
            .abortSignal(abortController.signal)
            .single()

        if (error && status !== 406) {
            // The status will be 406 if there are no results, or there are multiple results.
            // The single() method expects a single result.
            throw error;
        }

        return [user.id, data];
    } catch (error) {
        console.log('An error occurred while trying to retrieve the user profile.');
        console.log(error);
    }
    return [undefined, undefined];
}

/**
 * Retrieve the profile and id of the currently logged in user in realtime. The handler function
 * will be called on all data.
 */
export const getProfileRealTime = async (handler, abortController) => {
    const result = await getProfile(abortController);
    if (!result[0]) return;
    handler(result);

    try {
        const user = supabaseClient.auth.user();
        if (!user) {
            throw new Error('The profile couldn\'t be retrieved because the session data couldn\'t be read.');
        }

        return supabaseClient
            .from(`profiles:id=eq.${user.id}`)
            .on('*', (data) => handler([user.id, data.new]))
            .subscribe();

    } catch (error) {
        console.log('An error occurred while trying to retrieve the user profile.');
        console.log(error);
    }
    return undefined;
}

/**
 * Update or insert a new username into the user's profile.
 * @param username The new username that will be used to identify the user.
 * @return {Promise<boolean>} True if the request completed without error, false otherwise.
 */
export const upsertProfile = async (username) => {
    try {
        const user = supabaseClient.auth.user()

        const updates = {
            id: user.id,
            username,
            updated_at: new Date(),
        }

        let {error} = await supabaseClient
            .from('profiles')
            .upsert(updates, {
                returning: 'minimal', // Don't return the value after inserting
            })

        if (error) {
            throw error
        }
        return true;
    } catch (error) {
        console.log(`An error occurred while upserting a profile.`)
        console.error(error);
    }

    return false;
}

export const getUserId = () => {
    const userId = supabaseClient.auth.user()?.id;
    if (!userId) {
        throw new Error("User must be logged in to retrieve and edit notes.");
    }
    return userId;
}