import supabaseClient from './supabaseClient';
import {getUserId} from './userApi';
import {v4 as uuid} from 'uuid';

/**
 * Insert or update a note.
 * @param title
 * @param folder_id
 * @param id
 * @param content
 * @returns {Promise<{user_id: string, id: (*|string), title, folder_id, content: string}|undefined>}
 */
export const upsertNote = async (title, folder_id, id = undefined, content = undefined) => {
    try {
        const user_id = getUserId();

        const note = {
            title,
            folder_id,
            user_id,
            content: content || `# ${title}`,
            id: id || uuid()
        }

        const {error} = await supabaseClient
            .from('notes')
            .upsert(note, {returning: 'minimal'});

        if (error) {
            throw error;
        }

        return note;
    } catch (error) {
        console.log('An error occurred while upserting a note.')
        console.error(error);
    }
    return undefined;
}

export const getNote = async (noteId) => {
    // Promise added to illustrate the use of Suspense.
    // Comment to load faster.
    await new Promise(r => setTimeout(r, 1000));
    try {
        const {error, data} = await supabaseClient
            .from('notes')
            .select('*')
            .eq('id', noteId)
            .single();

        if (error) {
            throw error;
        }
        return data;
    } catch (error) {
        console.log(`Something went wrong while trying to retrieve the note with id ${noteId}`);
        console.error(error);
    }
    return undefined;
}

export const getNotesForDirectory = async (folderId) => {
    // Promise added to illustrate the use of Suspense.
    // Comment to load faster.
    await new Promise(r => setTimeout(r, 1000));
    try {
        const userId = getUserId();

        let result;
        if (folderId) {
            result = await supabaseClient
                .from('notes')
                .select('*')
                .eq('user_id', userId)
                .eq('folder_id', folderId);
        } else {
            result = await supabaseClient
                .from('notes')
                .select('*')
                .eq('user_id', userId)
                .is('folder_id', null);
        }
        const {error, data} = result;

        if (error) {
            throw error;
        }

        return data;
    } catch (error) {
        console.log(`An error occurred while retrieving the notes for the folder with id: ${folderId}`);
        console.error(error);
    }
    return [];
}

export const deleteNote = async (id) => {
    try {
        const {error} = await supabaseClient
            .from('notes')
            .delete({returning: 'minimal'})
            .eq('id', id);

        if (error) {
            throw error;
        }
        return true;
    } catch (error) {
        console.log(`Couldn't delete the note with id: ${id}`);
        console.error(error);
    }
    return false;
}

export const getNotesForUser = async () => {
    // await new Promise(r => setTimeout(r, 1500));
    try {
        const userId = getUserId();

        const {error, data} = await supabaseClient
            .from('notes')
            .select('*')
            .eq('user_id', userId);

        if (error) {
            throw error;
        }

        return data;
    } catch (error) {
        console.log(`An error occurred while retrieving the notes for the current user.`);
        console.error(error);
    }
    return [];
}
